// ─── SessionAlive Firefox Extension — Background Script ───

const API_BASE = "https://smsotp-2bc22.web.app/api";

// ─── Header Override State ───
// Stores active header overrides for session restore.
// Used by webRequest.onBeforeSendHeaders to modify outgoing request headers.
let activeHeaderOverride = null; // { rootDomain, headers: [{name, value}] }

// ─── Header Override via webRequest (blocking mode) ───
// Firefox MV2 supports blocking webRequest — we can modify headers directly.
// This replaces Chrome's declarativeNetRequest approach.

browser.webRequest.onBeforeSendHeaders.addListener(
  (details) => {
    if (!activeHeaderOverride) return;
    if (details.type !== "main_frame" && details.type !== "sub_frame") return;

    try {
      const hostname = new URL(details.url).hostname;
      if (!hostname.endsWith(activeHeaderOverride.rootDomain)) return;
    } catch (_) {
      return;
    }

    // Replace matching headers, keep the rest
    const overrideMap = new Map();
    for (const h of activeHeaderOverride.headers) {
      overrideMap.set(h.name.toLowerCase(), h.value);
    }

    const newHeaders = [];
    for (const h of details.requestHeaders) {
      const lower = h.name.toLowerCase();
      if (overrideMap.has(lower)) {
        newHeaders.push({ name: h.name, value: overrideMap.get(lower) });
        overrideMap.delete(lower);
      } else {
        newHeaders.push(h);
      }
    }
    // Add any overrides that weren't already present
    for (const [name, value] of overrideMap) {
      newHeaders.push({ name, value });
    }

    return { requestHeaders: newHeaders };
  },
  { urls: ["<all_urls>"] },
  ["blocking", "requestHeaders"]
);

// ─── Browser Headers Builder ───
// Builds complete browser fingerprint headers from navigator APIs.
// Produces the same header format as curl_cffi sessions.

function buildBrowserHeaders() {
  const ua = navigator.userAgent;
  const headers = {};

  headers["User-Agent"] = ua;

  // Firefox doesn't have navigator.userAgentData — derive from UA string
  const firefoxMatch = ua.match(/Firefox\/(\d+)/);
  const chromeMatch = ua.match(/Chrome\/(\d+)/);

  if (chromeMatch) {
    const ver = chromeMatch[1];
    headers["sec-ch-ua"] = `"Chromium";v="${ver}", "Google Chrome";v="${ver}", "Not-A.Brand";v="99"`;
    headers["sec-ch-ua-mobile"] = "?0";
  }
  if (firefoxMatch) {
    // Firefox doesn't send sec-ch-ua headers by default
  }

  let platform = "Unknown";
  if (/Windows/i.test(ua)) platform = "Windows";
  else if (/Macintosh|Mac OS/i.test(ua)) platform = "macOS";
  else if (/Linux/i.test(ua)) platform = "Linux";
  else if (/Android/i.test(ua)) platform = "Android";
  else if (/iPhone|iPad/i.test(ua)) platform = "iOS";
  headers["sec-ch-ua-platform"] = `"${platform}"`;

  headers["Upgrade-Insecure-Requests"] = "1";
  headers["Accept"] =
    "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,*/*;q=0.8";
  headers["Sec-Fetch-Site"] = "none";
  headers["Sec-Fetch-Mode"] = "navigate";
  headers["Sec-Fetch-User"] = "?1";
  headers["Sec-Fetch-Dest"] = "document";
  headers["Accept-Encoding"] = "gzip, deflate, br, zstd";

  if (navigator.languages && navigator.languages.length > 0) {
    const langs = navigator.languages.map((lang, i) => {
      if (i === 0) return lang;
      const q = Math.max(0.1, 1 - i * 0.1).toFixed(1);
      return `${lang};q=${q}`;
    });
    headers["Accept-Language"] = langs.join(",");
  } else {
    headers["Accept-Language"] = navigator.language || "en-US";
  }

  return headers;
}

// Detect browser version for impersonate field
function detectImpersonate(ua) {
  if (!ua) return "";
  const firefoxMatch = ua.match(/Firefox\/(\d+)/);
  if (firefoxMatch) return `firefox${firefoxMatch[1]}`;
  const chromeMatch = ua.match(/Chrome\/(\d+)/);
  if (chromeMatch) return `chrome${chromeMatch[1]}`;
  return "";
}

// ─── Helpers ───

async function getApiKey() {
  const result = await browser.storage.local.get(["apiKey"]);
  return result.apiKey || null;
}

async function apiRequest(method, path, body = null) {
  const apiKey = await getApiKey();
  if (!apiKey) throw new Error("Not authenticated");

  const options = {
    method,
    headers: {
      Authorization: `Bearer ${apiKey}`,
      "Content-Type": "application/json",
    },
  };
  if (body) options.body = JSON.stringify(body);

  const response = await fetch(`${API_BASE}${path}`, options);
  const data = await response.json();

  if (!response.ok) {
    throw new Error(data.error || `HTTP ${response.status}`);
  }
  return data;
}

// ─── Display Name Formatting ───

const MULTI_TLDS = new Set([
  "co.uk", "co.jp", "co.kr", "co.in", "co.nz", "co.za", "co.id",
  "com.au", "com.br", "com.cn", "com.mx", "com.sg", "com.tw", "com.hk",
  "org.uk", "org.au", "net.au", "ac.uk", "gov.uk",
]);

function extractDomainName(hostname) {
  const clean = hostname.replace(/^www\./, "");
  const parts = clean.split(".");
  if (parts.length <= 2) return parts[0];
  const lastTwo = parts.slice(-2).join(".");
  if (MULTI_TLDS.has(lastTwo) && parts.length > 2) return parts[parts.length - 3];
  return parts[parts.length - 2];
}

function formatDisplayName(hostname) {
  const name = extractDomainName(hostname);
  return name.charAt(0).toUpperCase() + name.slice(1);
}

// ─── Service Detection ───

function detectServiceFromUrl(url, cachedServices) {
  try {
    const parsed = new URL(url);
    const hostname = parsed.hostname.toLowerCase();

    // Reject non-HTTP URLs (about:newtab, about:blank, moz-extension://, etc.)
    if (!parsed.protocol.startsWith("http")) {
      return { detected: false };
    }

    if (cachedServices && cachedServices.length > 0) {
      // PASS 1: Precise domain matching (highest priority)
      // If a service has domains configured, match hostname against them exactly.
      for (const svc of cachedServices) {
        if (svc.domains && Array.isArray(svc.domains) && svc.domains.length > 0) {
          for (const d of svc.domains) {
            const domain = d.toLowerCase();
            if (hostname === domain || hostname.endsWith("." + domain)) {
              return {
                detected: true,
                serviceId: svc.id,
                baseUrl: `${parsed.protocol}//${parsed.hostname}`,
                displayName: svc.name || formatDisplayName(parsed.hostname),
                knownService: true,
              };
            }
          }
        }
      }

      // PASS 2: Keyword fallback — match service ID or name words in the hostname.
      // Handles services that don't have domains configured yet.
      const hostParts = hostname.split(".");
      for (const svc of cachedServices) {
        const keywords = new Set();
        keywords.add(svc.id.toLowerCase());
        if (svc.name) {
          for (const word of svc.name.toLowerCase().split(/[^a-z0-9]+/)) {
            if (word.length >= 3) keywords.add(word);
          }
        }
        for (const kw of keywords) {
          const matched = kw.length >= 4
            ? hostname.includes(kw)
            : hostParts.some((part) => part === kw);
          if (matched) {
            return {
              detected: true,
              serviceId: svc.id,
              baseUrl: `${parsed.protocol}//${parsed.hostname}`,
              displayName: svc.name || formatDisplayName(parsed.hostname),
              knownService: true,
            };
          }
        }
      }
    }

    // No match — generic (unknown service)
    const baseUrl = `${parsed.protocol}//${parsed.hostname}`;
    const serviceId = parsed.hostname.replace(/^www\./, "");
    const displayName = formatDisplayName(parsed.hostname);
    return { detected: true, serviceId, baseUrl, displayName, knownService: false };
  } catch (_) {}
  return { detected: false };
}

// ─── Root Domain Extraction ───

function getRootDomain(hostname) {
  const clean = hostname.replace(/^www\./, "");
  const parts = clean.split(".");
  if (parts.length <= 2) return clean;
  const lastTwo = parts.slice(-2).join(".");
  if (MULTI_TLDS.has(lastTwo) && parts.length > 2) {
    return parts.slice(-3).join(".");
  }
  return parts.slice(-2).join(".");
}

// ─── Cookie Capture ───
// Captures cookies belonging to the site's own domain hierarchy only.

async function getAllCookiesForTab(tabUrl) {
  let rootDomain;
  try {
    const parsed = new URL(tabUrl);
    rootDomain = getRootDomain(parsed.hostname);
  } catch (_) {
    return [];
  }

  const queries = [
    browser.cookies.getAll({ domain: rootDomain }).catch(() => []),
    browser.cookies.getAll({ domain: "." + rootDomain }).catch(() => []),
    browser.cookies.getAll({ url: tabUrl }).catch(() => []),
  ];

  const results = await Promise.all(queries);

  const seen = new Set();
  const allCookies = [];
  for (const batch of results) {
    for (const c of batch) {
      const cookieDomain = c.domain.replace(/^\./, "");
      if (!cookieDomain.endsWith(rootDomain)) continue;

      const key = `${c.domain}|${c.name}|${c.path}`;
      if (!seen.has(key)) {
        seen.add(key);
        allCookies.push(c);
      }
    }
  }

  return allCookies.map((c) => ({
    name: c.name,
    value: c.value,
    domain: c.domain,
    path: c.path,
    secure: c.secure,
    httpOnly: c.httpOnly,
    sameSite: c.sameSite,
    expirationDate: c.expirationDate || null,
  }));
}

// Simple cookie count for the current domain
async function getCookieCountForUrl(url) {
  try {
    const parsed = new URL(url);
    const cleanHost = parsed.hostname.replace(/^www\./, "");
    const queries = [
      browser.cookies.getAll({ url }).catch(() => []),
      browser.cookies.getAll({ domain: cleanHost }).catch(() => []),
      browser.cookies.getAll({ domain: "." + cleanHost }).catch(() => []),
    ];
    if (cleanHost !== parsed.hostname) {
      queries.push(browser.cookies.getAll({ domain: parsed.hostname }).catch(() => []));
    }
    const results = await Promise.all(queries);
    const seen = new Set();
    let count = 0;
    for (const batch of results) {
      for (const c of batch) {
        const key = `${c.domain}|${c.name}|${c.path}`;
        if (!seen.has(key)) {
          seen.add(key);
          count++;
        }
      }
    }
    return count;
  } catch (_) {
    return 0;
  }
}

// ─── Message Handler ───

browser.runtime.onMessage.addListener((message, _sender) => {
  // Return a Promise for async handling (Firefox supports this natively)
  return (async () => {
    try {
      switch (message.type) {
        case "detectService": {
          const stored = await browser.storage.local.get(["cachedServices"]);
          const services = stored.cachedServices || [];
          return detectServiceFromUrl(message.url, services);
        }

        case "getCookieCount": {
          const count = await getCookieCountForUrl(message.url);
          return { count };
        }

        case "saveCookies": {
          const cookies = await getAllCookiesForTab(message.tabUrl);
          if (cookies.length === 0) {
            return { success: false, error: "No cookies found for this site" };
          }

          const headers = buildBrowserHeaders();
          const ua = navigator.userAgent;
          const impersonate = detectImpersonate(ua);

          const data = await apiRequest("POST", "/extension/save-session", {
            service: message.serviceId,
            cookies,
            url: message.tabUrl,
            userAgent: ua,
            headers,
            impersonate,
          });

          // Return only minimal data to popup — no cookies/headers/UA
          return {
            success: data.success,
            sessionId: data.sessionId,
            email: data.email,
            service: data.service,
            serviceName: data.serviceName,
          };
        }

        case "validateApiKey": {
          await browser.storage.local.set({ apiKey: message.apiKey });
          try {
            const profile = await apiRequest("GET", "/profile");
            return { valid: true, ...profile };
          } catch (err) {
            await browser.storage.local.remove(["apiKey"]);
            return { valid: false, error: err.message };
          }
        }

        case "getProfile": {
          return await apiRequest("GET", "/profile");
        }

        case "getServices": {
          return await apiRequest("GET", "/services");
        }

        case "clearCookies": {
          try {
            const parsed = new URL(message.url);
            const rootDomain = getRootDomain(parsed.hostname);
            const sid = message.storeId || undefined;
            const getOpts = (q) => (sid ? { ...q, storeId: sid } : q);
            const batches = await Promise.all([
              browser.cookies.getAll(getOpts({ domain: rootDomain })).catch(() => []),
              browser.cookies.getAll(getOpts({ domain: "." + rootDomain })).catch(() => []),
              browser.cookies.getAll(getOpts({ url: message.url })).catch(() => []),
            ]);
            const seen = new Set();
            let removed = 0;
            for (const batch of batches) {
              for (const c of batch) {
                const key = `${c.domain}|${c.name}|${c.path}`;
                if (seen.has(key)) continue;
                seen.add(key);
                const domain = c.domain.replace(/^\./, "");
                const protocol = c.secure ? "https" : "http";
                const url = `${protocol}://${domain}${c.path}`;
                const removeOpts = { url, name: c.name };
                if (sid) removeOpts.storeId = sid;
                try {
                  await browser.cookies.remove(removeOpts);
                  removed++;
                } catch (_) {}
              }
            }
            return { success: true, removed };
          } catch (err) {
            return { success: false, error: err.message };
          }
        }

        case "setHeaderOverrides": {
          try {
            const hostname = new URL(message.url).hostname;
            const rootDomain = getRootDomain(hostname);
            const ua = message.userAgent;
            const sessionHeaders = message.headers || {};

            const overrideHeaders = [];

            if (ua) {
              overrideHeaders.push({ name: "User-Agent", value: ua });

              const isMobile = /Mobile|Android|iPhone|iPad/i.test(ua);
              overrideHeaders.push({
                name: "sec-ch-ua-mobile",
                value: isMobile ? "?1" : "?0",
              });

              let platform = "Unknown";
              if (/Windows/i.test(ua)) platform = "Windows";
              else if (/Macintosh|Mac OS/i.test(ua)) platform = "macOS";
              else if (/Linux/i.test(ua)) platform = "Linux";
              else if (/Android/i.test(ua)) platform = "Android";
              else if (/iPhone|iPad/i.test(ua)) platform = "iOS";
              overrideHeaders.push({
                name: "sec-ch-ua-platform",
                value: `"${platform}"`,
              });
            }

            for (const [key, val] of Object.entries(sessionHeaders)) {
              const lower = key.toLowerCase();
              if (lower === "user-agent" || lower === "cookie" || lower === "host") continue;
              if (val) overrideHeaders.push({ name: key, value: val });
            }

            activeHeaderOverride = overrideHeaders.length > 0
              ? { rootDomain, headers: overrideHeaders }
              : null;

            return { success: true, rules: overrideHeaders.length };
          } catch (err) {
            return { success: false, error: err.message };
          }
        }

        case "clearHeaderOverrides": {
          activeHeaderOverride = null;
          return { success: true };
        }

        case "getMySessions": {
          return await apiRequest("GET", "/extension/my-sessions");
        }

        case "restoreFromServer": {
          try {
            // Fetch full session from backend
            const data = await apiRequest("POST", "/extension/restore-session", {
              sessionId: message.sessionId,
              service: message.service,
            });

            if (!data.success || !data.session) {
              return { success: false, error: data.error || "Failed to fetch session" };
            }

            const session = data.session;
            const cookies = session.cookies || [];
            const targetUrl = session.baseUrl;

            if (!targetUrl) return { success: false, error: "Session has no base URL" };
            if (cookies.length === 0) return { success: false, error: "Session has no cookies" };

            const parsed = new URL(targetUrl);
            const rootDomain = getRootDomain(parsed.hostname);

            // Get cookie store ID
            let storeId;
            try {
              const wins = await browser.windows.getAll({ windowTypes: ["normal"] });
              if (wins.length > 0) {
                const tabs = await browser.tabs.query({ active: true, windowId: wins[wins.length - 1].id });
                if (tabs.length > 0) storeId = tabs[0].cookieStoreId;
              }
            } catch (_) {}

            const getOpts = (q) => (storeId ? { ...q, storeId } : q);

            // Clear existing cookies for this domain
            const clearBatches = await Promise.all([
              browser.cookies.getAll(getOpts({ domain: rootDomain })).catch(() => []),
              browser.cookies.getAll(getOpts({ domain: "." + rootDomain })).catch(() => []),
              browser.cookies.getAll(getOpts({ url: targetUrl })).catch(() => []),
            ]);
            const clearSeen = new Set();
            for (const batch of clearBatches) {
              for (const c of batch) {
                const key = `${c.domain}|${c.name}|${c.path}`;
                if (clearSeen.has(key)) continue;
                clearSeen.add(key);
                const domain = c.domain.replace(/^\./, "");
                const protocol = c.secure ? "https" : "http";
                const rmUrl = `${protocol}://${domain}${c.path}`;
                const rmOpts = { url: rmUrl, name: c.name };
                if (storeId) rmOpts.storeId = storeId;
                try { await browser.cookies.remove(rmOpts); } catch (_) {}
              }
            }

            // Set cookies from session
            let restored = 0;
            for (const c of cookies) {
              try {
                let cookieUrl;
                if (c.domain) {
                  const domain = c.domain.replace(/^\./, "");
                  const protocol = c.secure !== false ? "https" : "http";
                  cookieUrl = `${protocol}://${domain}${c.path || "/"}`;
                } else {
                  cookieUrl = targetUrl;
                }

                let sameSite = c.sameSite || "no_restriction";
                const isSecure = c.secure !== undefined ? c.secure : true;
                if (sameSite === "no_restriction" && !isSecure) sameSite = "lax";

                const details = {
                  url: cookieUrl,
                  name: c.name,
                  value: c.value,
                  path: c.path || "/",
                  secure: isSecure,
                  httpOnly: c.httpOnly !== undefined ? c.httpOnly : false,
                  sameSite,
                };
                if (c.domain) details.domain = c.domain;
                const expiry = c.expirationDate || c.expires;
                if (expiry) details.expirationDate = expiry;
                if (storeId) details.storeId = storeId;

                const result = await browser.cookies.set(details);
                if (result) restored++;
              } catch (_) {}
            }

            // Set header overrides for this domain
            const sessionUA = session.userAgent || "";
            const sessionHeaders = session.headers || {};
            if (sessionUA || Object.keys(sessionHeaders).length > 0) {
              const overrideHeaders = [];
              if (sessionUA) {
                overrideHeaders.push({ name: "User-Agent", value: sessionUA });
                const isMobile = /Mobile|Android|iPhone|iPad/i.test(sessionUA);
                overrideHeaders.push({ name: "sec-ch-ua-mobile", value: isMobile ? "?1" : "?0" });
                let platform = "Unknown";
                if (/Windows/i.test(sessionUA)) platform = "Windows";
                else if (/Macintosh|Mac OS/i.test(sessionUA)) platform = "macOS";
                else if (/Linux/i.test(sessionUA)) platform = "Linux";
                else if (/Android/i.test(sessionUA)) platform = "Android";
                else if (/iPhone|iPad/i.test(sessionUA)) platform = "iOS";
                overrideHeaders.push({ name: "sec-ch-ua-platform", value: `"${platform}"` });
              }
              for (const [key, val] of Object.entries(sessionHeaders)) {
                const lower = key.toLowerCase();
                if (lower === "user-agent" || lower === "cookie" || lower === "host") continue;
                if (val) overrideHeaders.push({ name: key, value: val });
              }
              activeHeaderOverride = overrideHeaders.length > 0
                ? { rootDomain, headers: overrideHeaders }
                : null;
            }

            // Return only display-safe data to popup
            return {
              success: true,
              url: targetUrl,
              serviceName: session.serviceName || session.service,
              email: session.email || "",
              restored,
              total: cookies.length,
            };
          } catch (err) {
            return { success: false, error: err.message };
          }
        }

        case "logout": {
          await browser.storage.local.remove(["apiKey"]);
          activeHeaderOverride = null;
          return { success: true };
        }

        default:
          return { error: "Unknown message type" };
      }
    } catch (err) {
      return { success: false, error: err.message };
    }
  })();
});
