// ─── SessionAlive Firefox Extension — Popup Script ───

const State = {
  apiKey: null,
  profile: null,
  services: [],
  currentTab: null,
  detectedService: null,
};

// ─── Helpers ───

function sendMessage(message) {
  return browser.runtime.sendMessage(message);
}

function showView(viewName) {
  document.querySelectorAll(".view").forEach((v) => v.classList.add("hidden"));
  document.getElementById(`view-${viewName}`).classList.remove("hidden");
}

function showState(stateName) {
  document.querySelectorAll("#service-card .state").forEach((s) => s.classList.add("hidden"));
  document.getElementById(`state-${stateName}`).classList.remove("hidden");
}

function showLoginError(msg) {
  const el = document.getElementById("login-error");
  el.textContent = msg;
  el.classList.remove("hidden");
}

function hideLoginError() {
  document.getElementById("login-error").classList.add("hidden");
}

function setLoginLoading(loading) {
  const btn = document.getElementById("login-btn");
  const text = document.getElementById("login-btn-text");
  const spinner = document.getElementById("login-spinner");
  if (loading) {
    btn.disabled = true;
    text.textContent = "Connecting...";
    spinner.classList.remove("hidden");
  } else {
    btn.disabled = false;
    text.textContent = "Connect";
    spinner.classList.add("hidden");
  }
}

// ─── User Info ───

function populateUserInfo() {
  const profile = State.profile;
  if (!profile) return;

  const email = profile.user?.email || "";
  const displayName = profile.user?.displayName || "";
  const nameToShow = displayName || email;
  const initial = nameToShow ? nameToShow.charAt(0).toUpperCase() : "U";
  const allocations = profile.sessions || {};
  const hasAllocations = Object.values(allocations).some((s) => s.allocated > 0);
  const isPaid = hasAllocations;
  const planLabel = isPaid ? "Paid" : "Free";

  document.getElementById("main-avatar").textContent = initial;
  document.getElementById("main-name").textContent = nameToShow;
  document.getElementById("main-email").textContent = email;
  document.getElementById("main-plan").textContent = planLabel;

  const extUsage = profile.extensionUsage || {};
  const saves = extUsage.saves || 0;
  const restores = extUsage.restores || 0;

  document.getElementById("counter-saves").textContent = isPaid
    ? `${saves}/unlimited`
    : `${saves}/10`;
  document.getElementById("counter-restores").textContent = isPaid
    ? `${restores}/unlimited`
    : `${restores}/10`;

  document.getElementById("counter-saves").className = isPaid
    ? "counter-value counter-unlimited"
    : "counter-value";
  document.getElementById("counter-restores").className = isPaid
    ? "counter-value counter-unlimited"
    : "counter-value";

  const allocEntries = Object.entries(allocations).filter(([, v]) => v.allocated > 0);
  const allocCard = document.getElementById("allocations-card");
  const allocList = document.getElementById("allocations-list");
  if (allocEntries.length > 0) {
    allocList.innerHTML = allocEntries
      .map(
        ([svcId, v]) =>
          `<div class="alloc-row">
            <span class="alloc-service">${v.serviceName || svcId}</span>
            <span class="alloc-count">${v.available}/${v.allocated}</span>
          </div>`
      )
      .join("");
    allocCard.classList.remove("hidden");
  } else {
    allocCard.classList.add("hidden");
  }

  if (State.apiKey) {
    document.getElementById("settings-api-key").textContent =
      State.apiKey.substring(0, 12) + "--------";
  }
}

// ─── Service Detection ───

async function detectService(url) {
  showState("detecting");

  const result = await sendMessage({ type: "detectService", url });

  if (result && result.detected) {
    State.detectedService = result;

    const cookieData = await sendMessage({ type: "getCookieCount", url });
    const count = cookieData?.count || 0;

    const backendService = State.services.find((s) => s.id === result.serviceId);
    const displayName =
      result.displayName || (backendService ? backendService.name : result.serviceId);
    const dotColor = backendService ? backendService.color : "#c8ff00";

    document.getElementById("service-name").textContent = displayName;
    document.getElementById("service-dot").style.background = dotColor;

    const saveBtn = document.getElementById("save-session-btn");
    const cookieInfo = document.getElementById("cookie-info");
    const noCookieHint = document.getElementById("no-cookie-hint");

    if (count > 0) {
      document.getElementById("cookie-count").textContent = count;
      cookieInfo.classList.remove("hidden");
      noCookieHint.classList.add("hidden");
      saveBtn.disabled = false;
      saveBtn.classList.remove("btn-disabled");
    } else {
      cookieInfo.classList.add("hidden");
      noCookieHint.classList.remove("hidden");
      saveBtn.disabled = true;
      saveBtn.classList.add("btn-disabled");
    }

    showState("detected");
  } else {
    showState("unsupported");
  }
}

// ─── Init ───

async function init() {
  showView("splash");
  const splashStart = Date.now();
  const SPLASH_MIN_MS = 1200;

  const stored = await browser.storage.local.get([
    "apiKey",
    "cachedProfile",
    "cachedServices",
  ]);

  if (!stored.apiKey) {
    await waitForSplash(splashStart, SPLASH_MIN_MS);
    showView("login");
    return;
  }

  State.apiKey = stored.apiKey;

  if (stored.cachedProfile) {
    State.profile = stored.cachedProfile;
    populateUserInfo();
  }

  if (stored.cachedServices) {
    State.services = stored.cachedServices;
  }

  if (!stored.cachedProfile) {
    await waitForSplash(splashStart, SPLASH_MIN_MS);
    showView("login");
    return;
  }

  const [tab] = await browser.tabs.query({ active: true, currentWindow: true });
  State.currentTab = tab;

  await waitForSplash(splashStart, SPLASH_MIN_MS);

  showView("main");
  if (tab?.url) {
    detectService(tab.url);
  } else {
    showState("unsupported");
  }

  refreshProfileInBackground();
}

function waitForSplash(startTime, minMs) {
  const elapsed = Date.now() - startTime;
  const remaining = minMs - elapsed;
  if (remaining <= 0) return Promise.resolve();
  return new Promise((resolve) => setTimeout(resolve, remaining));
}

async function refreshProfileInBackground() {
  try {
    const profile = await sendMessage({ type: "getProfile" });
    if (profile && !profile.error) {
      State.profile = profile;
      populateUserInfo();
      await browser.storage.local.set({ cachedProfile: profile });
    }
  } catch (_) {}

  try {
    const data = await sendMessage({ type: "getServices" });
    if (data?.services) {
      State.services = data.services;
      await browser.storage.local.set({ cachedServices: data.services });
    }
  } catch (_) {}
}

document.addEventListener("DOMContentLoaded", init);

// ─── Login Handler ───

document.getElementById("login-btn").addEventListener("click", async () => {
  const input = document.getElementById("api-key-input");
  const apiKey = input.value.trim();

  if (!apiKey) {
    showLoginError("Please enter your API key");
    return;
  }

  if (!apiKey.startsWith("sk_live_")) {
    showLoginError("Invalid key format. Keys start with sk_live_");
    return;
  }

  setLoginLoading(true);
  hideLoginError();

  try {
    const result = await sendMessage({ type: "validateApiKey", apiKey });

    if (!result || !result.valid) {
      throw new Error(result?.error || "Invalid API key");
    }

    State.apiKey = apiKey;
    State.profile = result;
    populateUserInfo();

    const data = await sendMessage({ type: "getServices" });
    State.services = data?.services || [];

    await browser.storage.local.set({
      cachedProfile: result,
      cachedServices: State.services,
    });

    const [tab] = await browser.tabs.query({ active: true, currentWindow: true });
    State.currentTab = tab;

    showView("main");
    if (tab?.url) {
      detectService(tab.url);
    } else {
      showState("unsupported");
    }
  } catch (err) {
    showLoginError(err.message);
  } finally {
    setLoginLoading(false);
  }
});

document.getElementById("api-key-input").addEventListener("keydown", (e) => {
  if (e.key === "Enter") {
    document.getElementById("login-btn").click();
  }
});

document.getElementById("toggle-key-visibility").addEventListener("click", () => {
  const input = document.getElementById("api-key-input");
  input.type = input.type === "password" ? "text" : "password";
});

// ─── Save Session ───

document.getElementById("save-session-btn").addEventListener("click", async () => {
  if (!State.detectedService || !State.currentTab) return;

  showState("saving");

  try {
    const response = await sendMessage({
      type: "saveCookies",
      tabUrl: State.currentTab.url,
      serviceId: State.detectedService.serviceId,
      baseUrl: State.detectedService.baseUrl,
    });

    if (!response || response.error || !response.success) {
      throw new Error(response?.error || "Unknown error");
    }

    const email = response.email || "unknown";
    const serviceName = response.serviceName || response.service;

    document.getElementById("success-message").textContent =
      `Session for ${email} saved to ${serviceName}.`;

    refreshProfileInBackground();
    showState("success");
  } catch (err) {
    document.getElementById("error-title").textContent = "Failed to save";
    document.getElementById("error-message").textContent = err.message;
    showState("error");
  }
});

// ─── Save Another ───

document.getElementById("save-another-btn").addEventListener("click", () => {
  if (State.currentTab?.url) {
    detectService(State.currentTab.url);
  }
});

// ─── Retry ───

document.getElementById("retry-btn").addEventListener("click", () => {
  if (State.currentTab?.url) {
    detectService(State.currentTab.url);
  }
});

// ─── Restore Session (inline) ───

function openRestoreView() {
  showView("restore");
  loadMySessions();
}

document.getElementById("restore-session-btn").addEventListener("click", openRestoreView);
document.getElementById("restore-session-btn-unsupported").addEventListener("click", openRestoreView);

// ─── Restore Window Logic ───

function showRestoreState(stateName) {
  document.querySelectorAll("#restore-card .state").forEach((s) => s.classList.add("hidden"));
  document.getElementById(`rw-state-${stateName}`).classList.remove("hidden");
}

let restoreResultUrl = null;
let allSessions = [];

function escapeHtml(str) {
  const div = document.createElement("div");
  div.textContent = str;
  return div.innerHTML;
}

function formatTimeAgo(isoString) {
  const now = Date.now();
  const then = new Date(isoString).getTime();
  const diffMs = now - then;
  const diffMins = Math.floor(diffMs / 60000);
  if (diffMins < 1) return "just now";
  if (diffMins < 60) return `${diffMins}m ago`;
  const diffHours = Math.floor(diffMins / 60);
  if (diffHours < 24) return `${diffHours}h ago`;
  const diffDays = Math.floor(diffHours / 24);
  if (diffDays < 7) return `${diffDays}d ago`;
  return new Date(isoString).toLocaleDateString();
}

function populateServiceFilter(sessions) {
  const select = document.getElementById("rw-service-filter");
  select.innerHTML = '<option value="all">All Services</option>';

  const serviceMap = new Map();
  for (const s of sessions) {
    if (!serviceMap.has(s.service)) {
      serviceMap.set(s.service, { name: s.serviceName, color: s.serviceColor || "#c8ff00", count: 0 });
    }
    serviceMap.get(s.service).count++;
  }

  for (const [svcId, info] of serviceMap) {
    const opt = document.createElement("option");
    opt.value = svcId;
    opt.textContent = `${info.name} (${info.count})`;
    select.appendChild(opt);
  }
}

function renderFilteredSessions() {
  const serviceFilter = document.getElementById("rw-service-filter").value;
  const emailQuery = document.getElementById("rw-email-search").value.trim().toLowerCase();
  const listEl = document.getElementById("rw-session-list");
  const countEl = document.getElementById("rw-session-count");
  const noMatchEl = document.getElementById("rw-no-match");

  let filtered = allSessions;

  if (serviceFilter !== "all") {
    filtered = filtered.filter((s) => s.service === serviceFilter);
  }

  if (emailQuery) {
    filtered = filtered.filter((s) => (s.email || "").toLowerCase().includes(emailQuery));
  }

  listEl.innerHTML = "";

  if (filtered.length === 0) {
    noMatchEl.classList.remove("hidden");
    countEl.textContent = "";
    return;
  }

  noMatchEl.classList.add("hidden");
  countEl.textContent = `${filtered.length} session${filtered.length !== 1 ? "s" : ""}`;

  for (const s of filtered) {
    const row = document.createElement("button");
    row.className = "session-row";

    const timeAgo = s.createdAt ? formatTimeAgo(s.createdAt) : "unknown";

    row.innerHTML = `
      <div class="session-row-dot" style="background: ${escapeHtml(s.serviceColor || "#c8ff00")}"></div>
      <div class="session-row-info">
        <span class="session-row-email">${escapeHtml(s.email || "No email")}</span>
        <span class="session-row-meta">${escapeHtml(s.serviceName)} &middot; ${s.cookiesCount} cookies &middot; ${timeAgo}</span>
      </div>
      <svg class="session-row-arrow" viewBox="0 0 20 20" fill="currentColor" width="14" height="14">
        <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"/>
      </svg>
    `;

    row.addEventListener("click", () => restoreSession(s.id, s.service));
    listEl.appendChild(row);
  }
}

async function loadMySessions() {
  showRestoreState("loading");

  // Reset filters
  document.getElementById("rw-service-filter").value = "all";
  document.getElementById("rw-email-search").value = "";

  try {
    const data = await sendMessage({ type: "getMySessions" });

    if (data?.error) {
      throw new Error(data.error);
    }

    const sessions = data?.sessions || [];

    if (sessions.length === 0) {
      showRestoreState("empty");
      return;
    }

    allSessions = sessions;
    populateServiceFilter(sessions);
    renderFilteredSessions();
    showRestoreState("pick");
  } catch (err) {
    document.getElementById("rw-error-message").textContent = err.message;
    showRestoreState("error");
  }
}

// Filter event listeners
document.getElementById("rw-service-filter").addEventListener("change", renderFilteredSessions);
document.getElementById("rw-email-search").addEventListener("input", renderFilteredSessions);

async function restoreSession(sessionId, service) {
  showRestoreState("restoring");

  try {
    const result = await sendMessage({
      type: "restoreFromServer",
      sessionId,
      service,
    });

    if (!result || !result.success) {
      throw new Error(result?.error || "Restore failed");
    }

    restoreResultUrl = result.url;

    document.getElementById("rw-success-message").textContent =
      `${result.restored}/${result.total} cookies restored for ${result.email || result.serviceName}.`;

    try {
      const siteName = new URL(result.url).hostname.replace(/^www\./, "");
      document.getElementById("rw-open-site-btn-text").textContent = `Open ${siteName}`;
    } catch (_) {
      document.getElementById("rw-open-site-btn-text").textContent = "Open Site";
    }

    showRestoreState("success");
  } catch (err) {
    document.getElementById("rw-error-message").textContent = err.message;
    showRestoreState("error");
  }
}

// "Open Site" — navigate the current tab
document.getElementById("rw-open-site-btn")?.addEventListener("click", async () => {
  if (!restoreResultUrl) return;
  try {
    if (State.currentTab) {
      await browser.tabs.update(State.currentTab.id, { url: restoreResultUrl });
    } else {
      await browser.tabs.create({ url: restoreResultUrl });
    }
  } catch (_) {
    await browser.tabs.create({ url: restoreResultUrl });
  }
  window.close();
});

// "Restore Another" button
document.getElementById("rw-restore-another-btn")?.addEventListener("click", () => {
  restoreResultUrl = null;
  loadMySessions();
});

// "Try Again" button
document.getElementById("rw-retry-btn")?.addEventListener("click", () => {
  restoreResultUrl = null;
  loadMySessions();
});

// ─── Back to Main ───

document.getElementById("restore-back-btn").addEventListener("click", () => {
  restoreResultUrl = null;
  showView("main");
  if (State.currentTab?.url) {
    detectService(State.currentTab.url);
  } else {
    showState("unsupported");
  }
});

// ─── Logout ───

document.getElementById("logout-btn").addEventListener("click", async () => {
  await sendMessage({ type: "logout" });
  await browser.storage.local.remove(["cachedProfile", "cachedServices"]);
  State.apiKey = null;
  State.profile = null;
  State.services = [];
  State.detectedService = null;
  document.getElementById("api-key-input").value = "";
  showView("login");
});

// ─── Copy API Key ───

document.getElementById("copy-key-btn").addEventListener("click", async () => {
  if (!State.apiKey) return;
  await navigator.clipboard.writeText(State.apiKey);
  const btn = document.getElementById("copy-key-btn");
  btn.title = "Copied!";
  setTimeout(() => {
    btn.title = "Copy";
  }, 2000);
});
